function [c,f,s] = hoffFK(x,t,u,DuDx,par)
% HOFFFK PDE components for the Hoffmann et al. equation
%
%   Example 1
%       No differentiation, only proliferation
%       ** Set r0 = 0 and r1 = 0 **
%       x = [0 0.05 0.09:0.01:0.21 0.25:0.05:1];
%       t = linspace(0,200,10);
%       mypde = @(x,t,u,DuDx) hoffFK(x,t,u,DuDx,[0 0]);
%       myic  = @(x) (x>=0.1).*(x<=0.2);
%       sol = pdepe(0,mypde,myic,@hoffFKbc,x,t);
%       u1 = sol(:,:,1);
%       surfc(x,t,u1,'LineStyle','none');
%
%   Example 2
%       Differentiation and proliferation
%       ** Set r0 = 0.01 and r1 = 0.5 **
%       x = 0:0.1:1;
%       t = linspace(0,200,10);
%       mypde = @(x,t,u,DuDx) hoffFK(x,t,u,DuDx,[0.01 0.05]);
%       myic  = @(x) (x>=0.1).*(x<=0.2);
%       sol = pdepe(0,mypde,myic,@hoffFKbc,x,t);
%       u1 = sol(:,:,1);
%       surfc(x,t,u1);
%
%   See also: HOFFFKBC, PDEPE
%
%   Ecole de Printemps SFBT 2012

maxProlif = 0.02;
r0 = par(1);
r1 = par(2);
maxNoise = 0.2;
minNoise = 0.1;

c = 1;
f = 0.5*(differentiation(x).*sigma2(x).*DuDx + ...
    (differentiation(x).*ds2(x)+ddr(x).*sigma2(x)).*u);
s = prolif(x)*u;

    function y = prolif(x)
        % PROLIF is the proliferation rate of x-cells        
        y = 4*maxProlif*x.*(1-x); 
    end

    function y = differentiation(x)
        % DIFFERENTIATION is the differentiation rate of x-cells     
        y = r0 + r1*prolif(x);
    end

    function y = ddr(x)
        % DDR derivative of DIFFERENTIATION
        % Used to compute f
        y = r1*4*maxProlif*(1-2*x);
    end

    function y = sigma2(x)
        % SIGMA2 differentiation noise
        y = minNoise + 4*maxNoise*(x-0.5).^2;
    end

    function y = ds2(x)
        % DS2 derivative of SIGMA2
        % Used to compute f
        y = 8*maxNoise*(x-0.5);
    end
end
