function [x,t,u1] = mscelldiff
% MSCELLDIFF Nonlinear PDE with population-dependent growth
%  MSCELLDIFF solve the modified Fokker-Planck equation adapted from
%  Hoffmann et al. (2008). The PDE is
%
%     [1] *  D_ [u] =  D_ [0.5*(R*s2*DuDx+(R*ds2+dR*s2)*u)] + [r(x)*u*(1-NA/K)]
%            Dt        Dx
%     ----      ---       ---------------------------------   ---------
%      c         u        f(x,t,u,Du/Dx)                      s(x,t,u,Du/Dx)
%
%   where
%       x           differentiation level x in [0,1]
%       t           time
%       u  = u(t,x) cell density at differentiation level x
%       R  = R(x)   differentiation rate
%       dR = dR(x)  derivative of the differentiation rate
%       s2 = s2(x)  variance of the differentiation jumps
%       ds2= ds2(x) derivative of s2
%       r  = r(x)   proliferation rate
%       NA =        cell population acting on the feedback
%       K  =        carrying capacity
%
%   Ecole de Printemps SFBT 2012

% model parameters are shared with the nested functions
maxProlif = 0.01;
r0 = 0.01;
r1 = 0.20;
maxNoise = 0.02;
minNoise = 0.005;
K = 0.05;



% spatial discretization and time span
nx = 101;
x0 = 0;
x1 = 1;
h = (x1-x0)/(nx-1);
x = 0:h:1;
tfinal = 2000;

% Population level parameters
Aset=find(x>=0.8);

% integration tolerance
utol=1e-1;
dtMin=1;
dtExpand=1.05;
dtShrink=0.95;
warning('on','MSCELLDIFF:integrationstep');


% init
% tic
tt=0; % current time
t=tt; % vector of all times
dt=10; % current time step
tspan = [tt tt+dt tt+2*dt];
cellsInA = trapz(x(Aset),hypdeic(x(Aset)));
sol = pdepe(0,@hypde,@hypdeic,@hybc,x,tspan);
u1(2:3,:) = sol(2:3,:,1);
i=2;
tt=tt+dt;
tspan = [tt tt+dt tt+2*dt];
newic = @(xout) pdeval(0,x,u1(i,:,1),xout);

% count the number of iterations
iterNb = 0;
progressCount = 0;
toprint = true;
fprintf('--------------------------------------------------\n')

% main loop
while tt<tfinal
    sol = pdepe(0,@hypde,newic,@hybc,x,tspan);
    % compare u1dt with u(i+2)
    u1old = u1(i+1,:);
    u1new = sol(2,:);
    u1(i+1:i+2,:) = sol(2:3,:,1);
    cellsInA = trapz(x(Aset),u1(i+1,Aset));
    if norm(u1new-u1old,2)<utol
        t(i:i+2) = tspan;
        dt=dtExpand*dt;
        i=i+1;
        tt=tt+dt;
    else        
        dt=max(dtShrink*dt,dtMin);
        if dt<=dtMin,
            warning('MSCELLDIFF:integrationstep','Minimun integration step dt reached, solutions will be inaccurate');
            warning('off','MSCELLDIFF:integrationstep');
        end        
    end
    tspan = [tt tt+dt tt+2*dt];
    newic = @(xout) pdeval(0,x,u1(i,:,1),xout);
    iterNb = iterNb+1;
    printProgress;
end
% toc 

fprintf('\n');

% return the number of iteration
% disp(['Number of iterations: ' num2str(iterNb)])

% plotting
surfc(x,t,u1,'EdgeColor',[0.8 0.8 0.9]);
set(gca,'FontS',14);
xlabel('differentiation alpha');
ylabel('time t');
zlabel('population density')
view(-159.5,44);
colormap jet
hold on
surf(x(Aset),t,-ones(length(t),length(x(Aset))));
plot3(x,0*x,prolif(x),'LineWidth',2,'Color',[0.8 0.4 0.4])
plot3(x,0*x,differentiation(x),'LineWidth',2,'Color',[0.8 0.4 0.8])
plot3(x,0*x,sigma2(x),'LineWidth',2,'Color',[0.8 0.8 0.4])
axis tight
hold off

    function [c,f,s] = hypde(x,~,u,DuDx)
        % HY PDE components for the Hoffmann-Yvinec equation
        c = 1;
        f = 0.5*(differentiation(x).*sigma2(x).*DuDx + ...
            (differentiation(x).*ds2(x)+ddr(x).*sigma2(x)).*u);
        s = prolif(x)*u*(1-cellsInA/K);
    end

    function u0 = hypdeic(x)
        % HYPDEIC Evaluates the Initial Conditions
        u0 = (x>=0.1).*(x<=0.2);
    end

    function [pl,ql,pr,qr] = hybc(~,~,~,~,~)
        % HYBC Boundary conditions
        pl = 0; pr = 0;
        ql = 1; qr = 1;
    end

    function y = prolif(x)
        % PROLIF is the proliferation rate of x-cells
        y = 4*maxProlif*x.*(1-x/0.8).*(x<0.8).*(x>0);
    end

    function y = differentiation(x)
        % DIFFERENTIATION is the differentiation rate of x-cells
        y = r0 + r1*prolif(x);
    end

    function y = ddr(x)
        % DDR derivative of DIFFERENTIATION
        % Used to compute f
        y = r1*4*maxProlif*(1-2*x);
    end

    function y = sigma2(x)
        % SIGMA2 differentiation noise
        y = minNoise + 4*maxNoise*(x-0.5).^2;
    end

    function y = ds2(x)
        % DS2 derivative of SIGMA2
        % Used to compute f
        y = 8*maxNoise*(x-0.5);
    end
    function printProgress
        % Print the progress of the solution
        if (tt > ((progressCount+1)*0.02*tfinal)) && toprint;
            fprintf('*');
            progressCount = progressCount+1;
            toprint = false;
        else
            toprint = true;
        end
        
    end
end
