function [x, long_run_cov, d] = varfima(N, d, cov_matrix, VAR, VMA, fivar, skip)

% Generates N observations of a realisation of a multivariate ARFIMA process X
% Let (e(t)) be a multivariate gaussian process with a covariance matrix cov_matrix.
% If fivar=TRUE the values of the process X are given by the equations:
%		VAR(L).U(t) = VMA(L).e(t)
%	 	diag((1-L)^d).X(t) = U(t)
% where L is the lag-operator.
% If fivar=FALSE the values of the process X are given by the equations:
%	 	diag((1-L)^d).U(t) = e(t)
%		VAR(L).X(t) = VMA(L).U(t)
%
% The vectorial ARMA operator is appied using function varima.sim of the package 'portes'
%
% INPUT     N 		Number of time points
%           d 		Vector of parameters of long-range dependence
%           cov_matrix    Matrix of correlation between the innovations (optional, default is identity)
%           VAR		Array of VAR coefficient matrices (optional)
%           VMA		Array of VMA coefficient matrices (optional)
%           fivar 	If TRUE, apply the ARMA operator before doing the fractional integration
%                   If FALSE, apply first the fractional integration before the ARMA operator
%                   Default is TRUE
%           skip	Number of initial observations omitted, after applying the ARMA operator 
%                   and the fractional integration (optional, default is 2000)
%         
% OUTPUT   x		Vector containing the N observations of the vector ARFIMA(arlags, d, malags) process
%	   long_run_cov	Matrix of covariance of the spectral density of x around the zero frequency
%	       d        Vector of parameters of long-range dependence, modified in case of cointegration
%
%										Achard & Gannaz (2014)
%__________________________________________________________________________________________________________________

k = length(d);

if(nargin<=3)
	VAR = [];
    VMA = [];
end
if(nargin<=4)
	VMA = [];
end
if(nargin<=5)
	fivar = 1;
end
if(nargin<=6)
	skip = 2000;
end

if(length(cov_matrix)~=k) 
	warning('Error, the size covariance matrix does not correspond to the number of long-range parameters.')
end

nn = N+2*skip;
r = chol(cov_matrix);


% tools to compute the long-run covariance matrix
[~, ~, nar] = size(VAR);
[~, ~, nma] = size(VMA);
Cvma = cov_matrix;
if(isempty(VMA)==0)	
	sumMA = 0;
	for lag=1:nma
		sumMA = sumMA+VMA(:,:,lag);
	end;
    	Cvma = (eye(k)+sumMA)*Cvma*(eye(k)+sumMA)';
end;
Cvar = eye(k);
if(isempty(VAR)==0)
	sumAR = 0;
	for lag=1:nar
		sumAR = sumAR+VAR(:,:,lag);
	end;
	Cvar = (eye(k)+sumAR)^-1;
end;


% simulate the data
e = randn(nn,k)*r;
if (fivar==1)
	u = varma(N+skip,k,VAR,VMA,cov_matrix,e);
   	long_run_cov = Cvar*Cvma*Cvar';
	x = fracdiff(u,d);
	x = x((length(x(:,1))-N+1):length(x(:,1)),:);
else
	u = fracdiff(e,d);
	u = u((skip+1):nn,:);
	x = varma(N,k,VAR,VMA,cov_matrix,u);
    long_run_cov = zeros(k);
    for l=1:k
        for m=1:k
            G = (Cvar(l,:)'*Cvar(m,:)).*Cvma;
            dG = (d'*ones(1,k)+ones(k,1)*d).*(G~=0);
            [temp, a] = max(abs(dG));
            [~, b] = max(temp);
            if (abs(dG(a(b),b))>=abs(dG(l,m)))
                long_run_cov(l,m) = G(a(b),b);
                if (l==m)
                    d(l) = (d(a(b))+d(b))/2;
                end;
            end
            
        end
    end
                
end


end

