function[magp]=magp(r,a,b,u,n)
%%% The function estimates lower and upper bound of the probability P(M(a,b)> u) 
%%% where M(a,b) is the maximum on the interval [a,b] of a centered Gaussian process X.
%%%
%%% Theoretical validity of these bounds are presented in Mercadier (2006).
%%% The lower bound is obtained by single discretization and the upper bound
%%% is obtained by combining an improvement of Rice's formula and a discretization. 
%%% 
%%% The estimation of these bounds is based on the MATLAB routine "rind" written by 
%%% Brodtkorb (2004) and available in the WAFO toolbox (http://www.maths.lth.se/matstat/wafo/).
%%% 
%%% REQUIREMENTS ON INPUT:
%%% r is the covariance of the process X satisfying regularity conditions imposed in Mercadier (2006);
%%% r is a two variable function whose variables have to be named by x and y;
%%% a and b are real numbers satisfying a < b;
%%% u is a real number;
%%% n is the number of discretized points taking into account in the bounds.
%%%
%%% EXEMPLE: 
%%% > syms x y;
%%% > r=exp(-(x-y)^2/2); %stationary case
%%% > magp(r,0,1,1,500) %returns 0.2533  0.2541
%%% > magp(r,0,1,2,500) %returns 0.0440  0.0442
%%% 
%%% > syms x y;
%%% > r=(exp(x*y)-1)/sqrt((exp(x^2)-1)*(exp(y^2)-1)); % non-stationary case
%%% > magp(r,1,2,1,500) %returns 0.2449  0.2449
%%% > magp(r,0,1,1,500) %returns 0.2299  0.2297
%%% Remark: in this last example the lower bound is greater than the upper bound.
%%% It is explained by the fact that both values may contain error estimation.
%%%
%%% REFERENCES:
%%% BRODTKORB, Per Andreas (2004)
%%% Numerical evaluation of multinormal expectations
%%% In Lund university report series
%%% and in the Dr.Ing thesis: 
%%% The probability of Occurrence of dangerous Wave Situations at Sea.
%%% Dr.Ing thesis, Norwegian University of Science and Technolgy, NTNU, Trondheim, Norway.
%%%
%%% MERCADIER, Ccile  (2006)
%%% Numerical bounds for the distribution of the maximum of one- and two-dimensional processes,
%%% Adv. in Appl. Probab. 38, no. 1, 149-170.

syms x y
covarfun=inline(vectorize(r));
ca=inline(vectorize(limit(covarfun(x,y),x,a,'right')));
va=limit(ca(y),y,a,'right');

lb=1-rind(covmatrix(r,a,b,n,1),zeros(n,1),-10^9,u,[0 n],[],n,rindoptset('method',1));
ub=1-wnormcdf(u,0,eval(va))+quad('magp_inner',a,b,[],[],r,a,u,n); 
magp=[lb,ub];
