//
// LiveCam.java
// LiveCam
//
// Created by Jochen Broz on 19.02.05.
// Copyright (c) 2005 Jochen Broz. All rights reserved.
// http://lists.apple.com/archives/quicktime-java/2005/Feb/msg00062.html
// Adapted by Christian Mercat 26.03.08 (resizable)
//

// For accessing the sequence grabber
import quicktime.*;
import quicktime.std.sg.*;
import quicktime.qd.*;
import quicktime.io.*;

// For creation of awt images
import java.awt.*;
import java.awt.image.*;
import javax.swing.*;



public class LiveCam {

int frameCount = 0;

// Data concerning the sequence grabber, its gWorld and its image size
SequenceGrabber sg;
QDRect cameraImageSize;
QDGraphics gWorld;


// Data concerning building awt images from cameras gWorld
public int[] pixelData;
public BufferedImage image;


/**
* flag, indicating that all capture and display tasks should continue or not
*/
boolean cameraActive = true;

/**
* Creates the LiveCam object.
*/
public LiveCam() throws Exception{
initSequenceGrabber(0,0);
initBufferedImage();
}
/**
* Creates the LiveCam object.
*/
public LiveCam(int width, int height) throws Exception{
initSequenceGrabber(width, height);
initBufferedImage();
}

/**
* Initializes the SequenceGrabber. Gets it's source video bounds, creates a gWorld with that size.
* Configures the video channel for grabbing, previewing and playing during recording.
*/
private void initSequenceGrabber(int width, int height) throws Exception{
sg = new SequenceGrabber();
SGVideoChannel vc = new SGVideoChannel(sg);
cameraImageSize = vc.getSrcVideoBounds();
if((width > 0) || (height > 0)) {
	cameraImageSize.resize( width, height);
	vc.setBounds(cameraImageSize);
}
	
gWorld =new QDGraphics(cameraImageSize);
sg.setGWorld(gWorld, null);

vc.setBounds(cameraImageSize);
vc.setUsage(quicktime.std.StdQTConstants.seqGrabRecord |
quicktime.std.StdQTConstants.seqGrabPreview |
quicktime.std.StdQTConstants.seqGrabPlayDuringRecord);
vc.setFrameRate(0);
vc.setCompressorType( quicktime.std.StdQTConstants.kComponentVideoCodecType);
}


/**
* This initializes the buffered image. First if determines the size of the data.
* Finds the number of ints per row, sets up the int array, where we can put the
* pixel data in. A DataBuffer is defined, using that int array. Based on this DataBuffer
* the BufferedImage is defined.
* The BufferedImage definition is not needed when the OpenGL view is used. There we only need
* the int-array pixelData. But I do not expect that defining the Buffered image has negative influence
* on OpenGLs performance.
*/
private void initBufferedImage() throws Exception{

int size = gWorld.getPixMap().getPixelData().getSize();
int intsPerRow = gWorld.getPixMap().getPixelData().getRowBytes()/4;

size = intsPerRow*cameraImageSize.getHeight();
pixelData = new int[size];
DataBuffer db = new DataBufferInt(pixelData, size);

ColorModel colorModel = new DirectColorModel(32, 0x00ff0000, 0x0000ff00, 0x000000ff);
int[] masks= {0x00ff0000, 0x0000ff00, 0x000000ff};
WritableRaster raster = Raster.createPackedRaster(db, cameraImageSize.getWidth(), cameraImageSize.getHeight(), intsPerRow, masks, null);
image = new BufferedImage(colorModel, raster, false, null);

}


/**
* This is a bit tricky. We do not start Previewing, but recording. By setting the output to a
* dummy file (which will never be created (hope so)) with the
* quicktime.std.StdQTConstants.seqGrabDontMakeMovie flag set. This seems to be equivalent to
* preview mode with the advantage, that it refreshes correctly.
*/
public void startPreviewing() throws Exception{
// QTFile movieFile = new QTFile(new java.io.File("NoFile"));
sg.setDataOutput( null, quicktime.std.StdQTConstants.seqGrabDontMakeMovie);
sg.prepare(true, true);
sg.startRecord();

// setting up a thread, to idle the sequence grabber
Runnable idleCamera = new Runnable(){
int taskingDelay = 25;
public void run(){
try{
while(cameraActive){
Thread.sleep(taskingDelay);
synchronized(sg){
sg.idleMore();
sg.update(null);
}
}
}catch(Exception ex){
ex.printStackTrace();
}
}
};
(new Thread(idleCamera)).start();
}


/**
* This creates a Panel, which displays the buffered image using awt. A Thread is started, copying
* the pixel data from the sequence grabbers gWorld to the data buffer of the BufferedImage. Then
* the image is repainted.
*/
public Component buildSwingCameraView(){
/**
* Used for the threading image update
*/
final Component ret = new Component(){
public void paint(Graphics g){
super.paint(g);
g.drawImage(image, 0, 0, this);
frameCount++;
};
};

Runnable imageUpdate = new Runnable(){
int taskingDelay = 10;
public void run(){
try{
while(cameraActive){
synchronized(sg){
gWorld.getPixMap().getPixelData().copyToArray(0, pixelData, 0, pixelData.length);
ret.repaint();
}
Thread.sleep(taskingDelay);
}
}catch(Exception ex){
ex.printStackTrace();
}
}
};
(new Thread(imageUpdate)).start();
return ret;
}



public QDRect getImageSize(){
return cameraImageSize;
}

public int getFrameCount(){
return frameCount;
}


public static void main (String args[]) {
try{
QTSession.open();

LiveCam myCam = new LiveCam(320,240);
JFrame cameraFrame = new JFrame("LiveCam");
myCam.startPreviewing();

// Switch between the two lines to select OpenGL or awt display:
Component imagePanel = myCam.buildSwingCameraView();
//Component imagePanel = myCam.buildOpenGLCameraView();

cameraFrame.add(imagePanel);
cameraFrame.setBounds(100, 100, myCam.getImageSize().getWidth(), myCam.getImageSize().getHeight());
cameraFrame.setVisible(true);

// Do some frame counting
System.out.println("Counting frames:");
Thread.sleep(5000);
int fc = myCam.getFrameCount();
Thread.sleep(10000);
System.out.println("Frame rate:"+0.1*(myCam.getFrameCount()-fc));

}catch(Exception ex){
ex.printStackTrace();
QTSession.close();
}

}
public boolean isCameraActive() {
	return cameraActive;
}
public void setCameraActive(boolean cameraActive) {
	this.cameraActive = cameraActive;
}
}
