// Copyright (C) 2013 - Michael Baudin
// Copyright (C) 2010 - DIGITEO - Michael Baudin
// Copyright (C) 1993 - 1995 - Anders Holtsberg
// Copyright (C) 1998 - 2000 - Mathematique Universite de Paris-Sud - Jean Coursol
//
// This file must be used under the terms of the CeCILL.
// This source file is licensed as described in the file COPYING, which
// you should have received as part of this distribution.  The terms
// are also available at
// http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt

function [h,edge]=histo(x,n,odd,scale,sym)
    // Plot a  histogram
    //  
    // Calling Sequence
    //   histo(x)
    //   histo(x)
    //   histo(x,n)
    //   histo(x,n,odd)
    //   histo(x,n,odd,scale)
    //   histo(x,n,odd,scale,sym)
    //   h=histo(...)
    //   [h,edge]=histo(...)
    //             
    // Parameters
    // x : datas
    // n : the approximate number of bins (default n = ceil(4*sqrt(sqrt(size(x,'*')))) ). If n is not a scalar, it contains the edges of the classes, in increasing order.
    // odd : set to 0 or 1 for placement of bins. Least significant digit of bin width will always be 1, 2 or 5. (default odd = 0)
    // scale : set to 1 to have the area 1 under the histogram instead of area n.  (default scale = 0 means no scaling)
    // sym : the integer (positive or negative) of the color of the plot. Negative is for a mark, positive is for a color.
    // egde : edges of the classes of the histogram
    // h : h(i) is the number of values in x that belong to [edge(i), edge(i+1)[ 
    //
    // Description
    // Compute and plots a histogram of the x.
    //
    // Any input argument equal to the empty matrix is replaced by its default value.
    //
    // The odd variable allows to configure the edges of the histogram.
    //
    // The advantage of the stixbox/histo function over the 
    // Scilab/histplot function is that the number of classes n can be automatically 
    // computed in histo, while it is the first argument of histplot.
    //
    // Examples
    // x=distfun_chi2rnd(3,1000,1);
    // scf();
    // histo(x);
    // xtitle("Chi-Square random numbers","X","Frequency")
    //
    // // Sets the approximate number of classes
    // scf();
    // histo(x,10);
    //
    // // Set the edges
    // X=distfun_unifrnd(0,1,1000,1);
    // edges = 0:0.2:1.; 
    // histo(X,edges);
    //
    // // See how the edges moves when we configure 
    // // the odd argument
    // scf(); 
    // histo(x,[],1);
    // scf(); 
    // histo(x,[],10);
    // scf(); 
    // histo(x,[],100);
    //
    // // See without scaling
    // scf(); 
    // histo(x,[],[],0);
    // // See with scaling
    // scf(); 
    // histo(x,[],[],1);
    //
    // // See various colors and styles
    // scf();
    // histo(x,[],[],[],1);
    // scf();
    // histo(x,[],[],[],2);
    // scf();
    // histo(x,[],[],[],3);
    //
    // Authors
    // Copyright (C) 2013 - Michael Baudin
    // Copyright (C) 2010 - DIGITEO - Michael Baudin
    // Copyright (C) 1993 - 1995 - Anders Holtsberg
    // Copyright (C) 1998 - 2000 - Mathematique Universite de Paris-Sud - Jean Coursol

    [nargout,nargin] = argn(0)

    if nargin < 2 then  
        n = [];
    end
    if nargin < 3 then  
        odd = [];
    end
    if nargin < 4 then  
        scale = [];
    end
    if nargin < 5 then  
        sym='b'; 
    end

    if n==[] then
        n = ceil(4*sqrt(sqrt(size(x,'*'))));
    end
    if odd==[] then
        odd = 0;
    end
    if scale==[] then
        scale = 0;
    end

    // Compute the limits
    if (size(n,"*")>1) then
        // n represents the edges.
        n=n(:)
        n=gsort(n,"g","i")
        limits=n'
    else
        mn = min(x);
        mx = max(x);
        d = (mx-mn)/n*2;
        e = floor(log(d)/log(10));
        m = floor(d/(10^e));
        if m>5 then
            m = 5;
        elseif m>2 then
            m = 2;
        end
        d = m*(10^e);
        mn = (floor(mn/d)-1)*d-odd*d/2;
        mx = (ceil(mx/d)+1)*d+odd*d/2;
        limits = mn:d:mx;
    end

    f = zeros(1,size(limits,'*')-1);
    for i = 1:size(limits,'*')-1
        f(i) = sum(bool2s((x>=limits(i))&(x<limits(i+1))),'m');
    end

    xx = [limits;limits;limits];
    xx = xx(:);
    xx = xx(2:$-1);
    yy = [f*0;f;f];
    yy = [yy(:);0];
    if scale then
        yy = yy/size(x,'*')/d;
    end
    // here we need sym -> scilab style 
    if nargin > 4 then
        plot2d(xx,yy,style=sym)
    else
        plot2d(xx,yy);
    end
    if nargout>0 
        h=f;
        edge=limits;
    end
endfunction


