// Copyright (C) 2013 - Michael Baudin
// Copyright (C) 2010 - DIGITEO - Michael Baudin
// Copyright (C) 1993 - 1995 - Anders Holtsberg
//
// This file must be used under the terms of the CeCILL.
// This source file is licensed as described in the file COPYING, which
// you should have received as part of this distribution.  The terms
// are also available at
// http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt

function [y,delta]=polyval(varargin)
    // Polynomial evaluation
    //
    // Calling Sequence
    //   y = polyval(p,x)
    //   y = polyval(p,x,S)
    //   [y,delta] = polyval(p,x,S)
    //
    // Parameters
    //   p : a (n+1)-by-1 matrix of doubles, the coefficients of the polynomial, with powers in decreasing order
    //   x : a nx-by-my matrix of doubles, the points where to evaluate the polynomial
    //   S : the optional data structure from polyfit
    //   y : a nx-by-my matrix of doubles, the value of the polynomial at x
    //   delta : 
    //
    // Description
    // If p is a vector of length n+1 whose elements are the coefficients of
    // a polynomial, then <literal>y=polyval(p,x)</literal> is the value of the polynomial,
    // defined by its coefficients p, evaluated at x.
    //
    // These coefficients are ordered with powers in decreasing order:
    //
    // <latex>
    // p(x) = p_1 x^n + p_2 x^{n-1} + ... + p_n x + p_{n+1}.
    // </latex>
    //
    // If x is a matrix, the polynomial is evaluated at all
    //    points in x.
    //
    // <literal>[y,delta] = polyval(p,x,S)</literal> uses the optional output generated by
    //    <literal>polyfit</literal> to generate error estimates, y+/-delta.
    //
    // Examples
    // p=[3 2 1];
    // y=polyval(p,[5 7 9])
    // expected=[86 162 262]
    //
    // // Evaluate for x matrix
    // x = linspace(0,%pi,10);
    // y = sin(x);
    // p = polyfit(x,y,3);
    // // Evaluate at x matrix
    // x = linspace(0,%pi,12);
    // x=matrix(x,3,4)
    // f=polyval(p,x)
    // y=sin(x)
    //
    // // Evaluate 95% confidence bounds, 
    // // i.e. +/- 2*delta
    // // Source: [1,2]
    // cdate=(1790:10:1990)';
    // pop=[
    //     3.929214   
    //     5.308483   
    //     7.239881   
    //     9.638453   
    //     12.860702  
    //     17.063353  
    //     23.191876  
    //     31.443321  
    //     38.558371  
    //     50.189209  
    //     62.979766  
    //     76.212168  
    //     92.228496  
    //     106.02154  
    //     123.20262  
    //     132.16457  
    //     151.3258   
    //     179.32317  
    //     203.30203  
    //     226.5422   
    //     248.70987  
    //  ];
    // scf();
    // plot(cdate,pop,"+")
    // // Calculate fit parameters
    // [p,S] = polyfit(cdate,pop,2);
    // // Evaluate the fit and the prediction error estimate (delta)
    // [pop_fit,delta] = polyval(p,cdate,S);
    // // Plot the data, the fit, and the confidence bounds
    // plot(cdate,pop_fit,"g-")
    // plot(cdate,pop_fit+2*delta,"r:")
    // plot(cdate,pop_fit-2*delta,"r:")
    // // Annotate the plot
    // xlabel("Census Year");
    // ylabel("Population (millions)");
    // title("Quadratic Polynomial Fit with Confidence Bounds")
    // 
    // Authors
    // Copyright (C) 2013 - Michael Baudin
    // Copyright (C) 2010 - DIGITEO - Michael Baudin
    // Copyright (C) 1993 - 1995 - Anders Holtsberg
    //
    // Bibliography
    // [1] http://en.wikipedia.org/wiki/Polynomial_interpolation
    // [2] http://www.mathworks.fr/fr/help/matlab/data_analysis/programmatic-fitting.html
    // [3] Numerical Computing with MATLAB, Cleve Moler, 2004
    // 

    [lhs,rhs]=argn()
    apifun_checkrhs ( "polyval" , rhs , 2:3 )
    apifun_checklhs ( "polyval" , lhs , 1:2 )
    //
    p = varargin ( 1 )
    x = varargin ( 2 )
    S = apifun_argindefault(varargin,3,[])
    //
    // Check Type
    apifun_checktype ( "polyval" , p , "p" , 1 , "constant" )
    apifun_checktype ( "polyval" , x , "x" , 2 , "constant" )
    if (S<>[]) then
        apifun_checktype ( "polyval" , S , "S" , 3 , "st" )
    end
    //
    // Check Size
    // TODO
    //
    // Check content
    expfields=[
    "R"
    "df"
    "normr"
    ]
    if (S<>[]) then
    if (fieldnames(S)<>expfields) then
        error(msprintf(gettext("%s: Wrong input argument #%d: Expected output argument of polyfit"),"polyval",3))
    end
    end
    //
    y=[];
    delta=[];
    [nargout,nargin] = argn(0)

    [m,n] = size(x);
    nc = max(size(p));

    if (m+n)==2 then
        // Make it scream for scalar x.  Polynomial evaluation can be
        // implemented as a recursive digital filter.
        y = mtlb_filter(1,[1,-x],p);
        y = y(nc);
        if rhs<3 then
            return

        end
    end

    // Do general case where X is an array
    y = zeros(m,n);
    for i = 1:nc
        y = x .* y+p(i);
    end

    if (rhs>2)&(nargout>1) then
        // Compute delta
        x = x(:);
        R=S.R
        degreef=S.df
        normr=S.normr
        // Construct Vandermonde matrix.
        V(:,nc) = ones(mtlb_length(x),1);
        for j = nc-1:-1:1
            V(:,j) = x .* V(:,j+1);
        end
        // Workaround for bug #9196
        // http://bugzilla.scilab.org/show_bug.cgi?id=9196
        // E = V/R;
        E = linalg_dgesv(R',V')'
        if nc==1 then
            e = sqrt(1+E .* E);
        else
            e = sqrt(1+sum(E.^2,"c")');
        end
        if degreef==0 then
            warning(msprintf(gettext("%s: zero degrees of freedom implies infinite error bounds."),"polyval"));
            delta = %inf*e;
        else
            delta = e*normr/sqrt(degreef)
        end
        delta = matrix(delta,m,n);
    end

endfunction
